<?php
/**
 * Plugin Name: Customer Feedback to Notion
 * Description: Collects customer feedback after purchase (buying process, page rating, speed) and sends data to Notion database
 * Version: 1.0.3
 * Author: Istvan Barcza
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * WC requires at least: 3.0
 * WC tested up to: 8.0
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

add_action('before_woocommerce_init', function() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

class CustomerFeedbackToNotion {

    private static $instance = null;

    // Constants for configuration
    const TOKEN_EXPIRY_DAYS = 5;  // Default token expiry in days
    const RATE_LIMIT_MAX_SUBMISSIONS = 3;
    const RATE_LIMIT_TIME_WINDOW = 300; // 5 minutes in seconds
    const MAX_RATING_VALUE = 5;
    const MIN_RATING_VALUE = 1;
    const MAX_COMMENT_LENGTH = 2000;
    const MAX_REDIRECT_DELAY = 30;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }
    
    public function init() {
        add_action('woocommerce_order_status_completed', array($this, 'send_feedback_email'), 10, 1);
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_submit_customer_feedback', array($this, 'handle_feedback_submission'));
        add_action('wp_ajax_nopriv_submit_customer_feedback', array($this, 'handle_feedback_submission'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_cftn_test_notion_connection', array($this, 'test_notion_connection'));
        add_action('wp_ajax_cftn_sync_feedback_data', array($this, 'sync_feedback_data'));
        add_action('wp_ajax_cftn_clear_error', array($this, 'clear_last_error'));
        add_shortcode('customer_feedback_form', array($this, 'feedback_form_shortcode'));

        // Security: Add Content Security Policy headers ONLY on feedback pages
        add_action('wp_head', array($this, 'add_security_headers'));

        // Handle feedback token URLs - use multiple hooks for reliability
        add_action('init', array($this, 'early_feedback_check'), 5);
        add_action('template_redirect', array($this, 'handle_feedback_redirect'), 1);
        add_filter('query_vars', array($this, 'add_query_vars'));
        
        $this->create_feedback_table();
    }
    
    public function activate() {
        if (!wp_next_scheduled('cftn_cleanup_expired_tokens')) {
            wp_schedule_event(time(), 'daily', 'cftn_cleanup_expired_tokens');
        }
        $this->create_feedback_table();
        flush_rewrite_rules();
        
        add_action('cftn_cleanup_expired_tokens', array($this, 'cleanup_expired_tokens'));
    }
    
    public function cleanup_expired_tokens() {
        global $wpdb;

        // Clean up expired feedback tokens
        $options = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('cftn_token_') . '%'
            )
        );

        foreach ($options as $option) {
            $token_data = get_option($option->option_name);
            if ($token_data && isset($token_data['expires']) && $token_data['expires'] < time()) {
                delete_option($option->option_name);
            }
        }

        // Clean up expired rate limit transients (WordPress auto-cleans, but we do it explicitly)
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                 WHERE option_name LIKE %s
                 AND option_name LIKE %s",
                $wpdb->esc_like('_transient_timeout_cftn_rate_limit_') . '%',
                '%'
            )
        );

        // Remove the associated transient values for expired timeouts
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                 WHERE option_name LIKE %s
                 AND option_name NOT IN (
                     SELECT CONCAT('_transient_', SUBSTRING(option_name, 20))
                     FROM {$wpdb->options}
                     WHERE option_name LIKE %s
                 )",
                $wpdb->esc_like('_transient_cftn_rate_limit_') . '%',
                $wpdb->esc_like('_transient_timeout_') . '%'
            )
        );
    }

    public function add_settings_link($links) {
        $settings_link = '<a href="' . esc_url(admin_url('admin.php?page=customer-feedback-notion-settings')) . '">' . __('Settings', 'customer-feedback-to-notion') . '</a>';
        $plugin_page_link = '<a href="' . esc_url('https://cood3.com/ctnf/') . '" target="_blank" rel="noopener noreferrer"><strong>' . __('Plugin page', 'customer-feedback-to-notion') . '</strong></a>';
        array_unshift($links, $settings_link);
        array_push($links, $plugin_page_link);
        return $links;
    }
    
    private function create_feedback_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            order_id bigint(20) NOT NULL,
            customer_email varchar(100) NOT NULL,
            customer_name varchar(100) NOT NULL,
            buying_process_rating tinyint(1) NOT NULL,
            page_rating tinyint(1) NOT NULL,
            speed_rating tinyint(1) NOT NULL,
            additional_comments text,
            submitted_at datetime DEFAULT CURRENT_TIMESTAMP,
            notion_synced tinyint(1) DEFAULT 0,
            notion_page_id varchar(100),
            PRIMARY KEY (id),
            UNIQUE KEY order_id (order_id),
            KEY customer_email (customer_email),
            KEY submitted_at (submitted_at),
            KEY notion_synced (notion_synced)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function enqueue_scripts() {
        // Load scripts if feedback_token is present (any page) or on pages with the shortcode
        if (get_query_var('feedback_token') || (is_page() && has_shortcode(get_post()->post_content, 'customer_feedback_form'))) {
            wp_enqueue_script('cftn-feedback', plugin_dir_url(__FILE__) . 'assets/feedback.js', array('jquery'), '1.0.0', true);
            wp_localize_script('cftn-feedback', 'cftn_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('cftn_feedback_nonce')
            ));
            wp_enqueue_style('cftn-feedback', plugin_dir_url(__FILE__) . 'assets/feedback.css', array(), '1.0.0');
        }
    }
    
    public function send_feedback_email($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Check if feedback already exists for this order
        global $wpdb;
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        $existing_feedback = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table_name} WHERE order_id = %d",
            $order_id
        ));

        if ($existing_feedback) {
            // Feedback already submitted for this order, don't send email again
            return;
        }

        $customer_email = $order->get_billing_email();
        $customer_name = $order->get_billing_first_name();

        // Get token expiry days from settings (default to constant if not set)
        $expiry_days = get_option('cftn_token_expiry_days', self::TOKEN_EXPIRY_DAYS);
        if ($expiry_days < 1 || $expiry_days > 30) {
            $expiry_days = self::TOKEN_EXPIRY_DAYS; // Fallback to default if invalid
        }

        // Use cryptographically secure random token
        $token = bin2hex(random_bytes(32));
        $expires = time() + ($expiry_days * DAY_IN_SECONDS);

        update_option("cftn_token_{$token}", array(
            'order_id' => $order_id,
            'customer_email' => $customer_email,
            'expires' => $expires
        ));

        // Use proper WordPress URL with query parameter
        $feedback_url = add_query_arg('feedback_token', $token, home_url('/'));

        // Get custom email subject and body from settings
        $default_subject = __('How was your shopping experience?', 'customer-feedback-to-notion');
        $default_body = "Hi {customer_name},\n\nThank you for your recent order #{order_number}!\n\nWe'd love to hear about your shopping experience. Please take a moment to share your feedback:\n\n{feedback_url}\n\nThis link will expire in {expiry_days} days.\n\nThank you!";

        $subject = get_option('cftn_email_subject', $default_subject);
        $message = get_option('cftn_email_body', $default_body);

        // Replace placeholders in subject
        $subject = str_replace(
            array('{customer_name}', '{order_number}', '{feedback_url}', '{expiry_days}'),
            array($customer_name, $order->get_order_number(), $feedback_url, $expiry_days),
            $subject
        );

        // Replace placeholders in message
        $message = str_replace(
            array('{customer_name}', '{order_number}', '{feedback_url}', '{expiry_days}'),
            array($customer_name, $order->get_order_number(), $feedback_url, $expiry_days),
            $message
        );

        // Allow filtering of email subject and message
        $subject = apply_filters('cftn_feedback_email_subject', $subject, $order, $token);
        $message = apply_filters('cftn_feedback_email_message', $message, $order, $token, $feedback_url);

        // Set proper email headers
        $headers = array(
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>',
            'Reply-To: ' . get_option('admin_email')
        );

        // Allow filtering of headers
        $headers = apply_filters('cftn_feedback_email_headers', $headers, $order, $token);

        wp_mail($customer_email, $subject, $message, $headers);
    }
    
    public function feedback_form_shortcode($atts) {
        $token = get_query_var('feedback_token');
        if (!$token) {
            return '<p>' . __('Invalid feedback link.', 'customer-feedback-to-notion') . '</p>';
        }
        
        $token_data = get_option("cftn_token_{$token}");
        if (!$token_data || $token_data['expires'] < time()) {
            return '<p>' . __('This feedback link has expired.', 'customer-feedback-to-notion') . '</p>';
        }
        
        ob_start();
        ?>
        <div id="cftn-feedback-form">
            <h3><?php _e('How was your shopping experience?', 'customer-feedback-to-notion'); ?></h3>
            <form id="customer-feedback-form">
                <div class="cftn-field">
                    <label><?php _e('How would you rate the buying process?', 'customer-feedback-to-notion'); ?></label>
                    <div class="cftn-rating">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <input type="radio" name="buying_process" value="<?php echo $i; ?>" id="process_<?php echo $i; ?>">
                            <label for="process_<?php echo $i; ?>">⭐</label>
                        <?php endfor; ?>
                    </div>
                </div>
                
                <div class="cftn-field">
                    <label><?php _e('How would you rate our website/page?', 'customer-feedback-to-notion'); ?></label>
                    <div class="cftn-rating">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <input type="radio" name="page_rating" value="<?php echo $i; ?>" id="page_<?php echo $i; ?>">
                            <label for="page_<?php echo $i; ?>">⭐</label>
                        <?php endfor; ?>
                    </div>
                </div>
                
                <div class="cftn-field">
                    <label><?php _e('How would you rate the website speed?', 'customer-feedback-to-notion'); ?></label>
                    <div class="cftn-rating">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <input type="radio" name="speed_rating" value="<?php echo $i; ?>" id="speed_<?php echo $i; ?>">
                            <label for="speed_<?php echo $i; ?>">⭐</label>
                        <?php endfor; ?>
                    </div>
                </div>
                
                <div class="cftn-field">
                    <label for="additional_comments"><?php _e('Additional comments (optional):', 'customer-feedback-to-notion'); ?></label>
                    <textarea name="additional_comments" id="additional_comments" rows="4"></textarea>
                </div>
                
                <input type="hidden" name="feedback_token" value="<?php echo esc_attr($token); ?>">
                <button type="submit"><?php _e('Submit Feedback', 'customer-feedback-to-notion'); ?></button>
            </form>
            <div id="cftn-message"></div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get real client IP address (handles proxies)
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR');

        foreach ($ip_keys as $key) {
            if (isset($_SERVER[$key]) && !empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];

                // Handle comma-separated IPs (from X-Forwarded-For)
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }

                // Validate IP
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return sanitize_text_field($ip);
                }
            }
        }

        return sanitize_text_field($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
    }

    /**
     * Rate limiting check for form submissions
     */
    private function check_rate_limit($max_submissions = null, $time_window = null) {
        if ($max_submissions === null) {
            $max_submissions = self::RATE_LIMIT_MAX_SUBMISSIONS;
        }
        if ($time_window === null) {
            $time_window = self::RATE_LIMIT_TIME_WINDOW;
        }

        $user_ip = $this->get_client_ip();
        
        // Create a unique key for this IP
        $rate_limit_key = 'cftn_rate_limit_' . md5($user_ip);
        
        // Get current submissions for this IP
        $submissions = get_transient($rate_limit_key);
        if (!$submissions) {
            $submissions = array();
        }
        
        // Clean up old submissions outside the time window
        $current_time = time();
        $submissions = array_filter($submissions, function($timestamp) use ($current_time, $time_window) {
            return ($current_time - $timestamp) < $time_window;
        });
        
        // Check if limit is exceeded
        if (count($submissions) >= $max_submissions) {
            return false; // Rate limit exceeded
        }
        
        // Record this submission attempt
        $submissions[] = $current_time;
        set_transient($rate_limit_key, $submissions, $time_window);
        
        return true; // Rate limit OK
    }
    
    /**
     * Security audit logging (only when WP_DEBUG is enabled)
     */
    private function log_security_event($action, $details = array()) {
        // Only log in debug mode to avoid performance impact in production
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            return;
        }

        $user_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

        $log_entry = array(
            'timestamp' => current_time('mysql'),
            'ip_address' => sanitize_text_field($user_ip),
            'action' => $action,
            'details' => $details
        );

        error_log('CFTN_AUDIT: ' . json_encode($log_entry));
    }
    
    /**
     * Add security headers including CSP - ONLY on feedback pages
     */
    public function add_security_headers() {
        // Only apply security headers on feedback form pages, not on the entire site
        $is_feedback_page = get_query_var('feedback_token') ||
                           (isset($_GET['feedback_token']) && !empty($_GET['feedback_token'])) ||
                           (is_page() && has_shortcode(get_post()->post_content ?? '', 'customer_feedback_form'));

        if (!$is_feedback_page) {
            return; // Don't interfere with other pages (like WooCommerce)
        }

        // Content Security Policy - only for feedback pages
        $csp = "default-src 'self'; " .
               "script-src 'self' 'unsafe-inline' https://www.google-analytics.com https://connect.facebook.net; " .
               "style-src 'self' 'unsafe-inline'; " .
               "img-src 'self' data: https:; " .
               "connect-src 'self' https://api.notion.com https://www.google-analytics.com; " .
               "frame-ancestors 'none'; " .
               "form-action 'self';";

        // Only add if not already set
        if (!headers_sent()) {
            header("Content-Security-Policy: $csp");
            header("X-Frame-Options: SAMEORIGIN");
            header("X-Content-Type-Options: nosniff");
            header("X-XSS-Protection: 1; mode=block");
            header("Referrer-Policy: strict-origin-when-cross-origin");
        }

        // Also add as meta tag for backup
        echo '<meta http-equiv="Content-Security-Policy" content="' . esc_attr($csp) . '">' . PHP_EOL;
        echo '<meta http-equiv="X-Frame-Options" content="SAMEORIGIN">' . PHP_EOL;
    }
    
    /**
     * Add feedback_token to query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'feedback_token';
        return $vars;
    }
    
    /**
     * Early check for feedback token in URL - catches it before normal WordPress processing
     */
    public function early_feedback_check() {
        // Check if this is a feedback token request
        if (isset($_GET['feedback_token']) && !empty($_GET['feedback_token'])) {
            $token = sanitize_text_field($_GET['feedback_token']);

            // Validate token
            $token_data = get_option("cftn_token_" . $token);
            if (!$token_data || $token_data['expires'] < time()) {
                wp_redirect(home_url('/?feedback_error=expired'));
                exit;
            }
        }
    }
    
    /**
     * Handle feedback token redirects
     */
    public function handle_feedback_redirect() {
        // Check multiple ways to get the token
        $token = get_query_var('feedback_token');
        
        // Fallback: check $_GET directly if query_var doesn't work
        if (!$token && isset($_GET['feedback_token'])) {
            $token = sanitize_text_field($_GET['feedback_token']);
        }

        if ($token) {
            // Validate token
            $token_data = get_option("cftn_token_" . $token);
            if (!$token_data || $token_data['expires'] < time()) {
                wp_redirect(home_url('/?feedback_error=expired'));
                exit;
            }

            // Load the feedback form template
            $this->load_feedback_template($token);
            exit;
        }
    }
    
    /**
     * Load feedback form template
     */
    private function load_feedback_template($token) {
        // Set up the global query var so the shortcode can access it
        global $wp_query;
        if ($wp_query) {
            $wp_query->set('feedback_token', $token);
        }
        
        // Status header to prevent caching
        status_header(200);
        header('Cache-Control: no-cache, no-store, must-revalidate');
        
        // Start output buffer to capture the page
        ob_start();
        
        // Basic HTML structure if theme functions aren't available
        if (!function_exists('get_header')) {
            echo '<!DOCTYPE html><html><head>';
            echo '<title>Customer Feedback Form</title>';
            echo '<meta charset="UTF-8">';
            echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
            echo '</head><body>';
        } else {
            get_header();
        }
        
        // Add CSS (always include, regardless of theme)
        echo '<style>
        .cftn-feedback-page {
            max-width: 800px;
            margin: 2rem auto;
            padding: 2rem;
            background: #fff;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border-radius: 8px;
        }
        .cftn-feedback-page h3 {
            text-align: center;
            margin-bottom: 2rem;
            color: #333;
        }
        .cftn-field {
            margin-bottom: 1.5rem;
        }
        .cftn-rating {
            display: flex;
            gap: 5px;
            margin: 10px 0;
        }
        .cftn-rating input[type="radio"] {
            display: none;
        }
        .cftn-rating label {
            font-size: 28px;
            cursor: pointer;
            color: #ddd;
            transition: color 0.2s;
        }
        .cftn-rating label:hover {
            color: #ffb400;
        }
        .cftn-rating input:checked + label,
        .cftn-rating.selected label.selected {
            color: #ffc107;
        }
        </style>';
        
        // Load the feedback form
        echo '<div class="cftn-feedback-page">';
        
        // Manually call the shortcode function to ensure it works
        echo $this->feedback_form_shortcode(array());
        
        echo '</div>';

        // Enqueue scripts properly (no more inline JavaScript)
        wp_enqueue_script('jquery');
        wp_enqueue_script('cftn-feedback-template', plugin_dir_url(__FILE__) . 'assets/feedback-template.js', array('jquery'), '1.0.3', true);

        // Localize script with necessary data
        wp_localize_script('cftn-feedback-template', 'cftn_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cftn_feedback_nonce'),
            'feedback_token' => $token
        ));

        // Print enqueued scripts in footer
        wp_print_scripts();

        // Close HTML structure
        if (!function_exists('get_footer')) {
            echo '</body></html>';
        } else {
            get_footer();
        }
        
        // Output the buffer
        ob_end_flush();
    }
    
    public function handle_feedback_submission() {
        // Validate POST parameters exist
        if (!isset($_POST['nonce']) || !isset($_POST['feedback_token'])) {
            wp_send_json_error('Missing required parameters');
        }

        if (!wp_verify_nonce($_POST['nonce'], 'cftn_feedback_nonce')) {
            wp_die('Security check failed');
        }

        // Rate limiting: Check if user has submitted too many feedbacks recently
        if (!$this->check_rate_limit()) {
            wp_die('Too many submissions. Please wait before submitting again.');
        }

        $token = sanitize_text_field($_POST['feedback_token']);
        $token_data = get_option("cftn_token_{$token}");

        if (!$token_data || $token_data['expires'] < time()) {
            wp_send_json_error('Invalid or expired token');
        }

        // Check if feedback already exists for this order
        global $wpdb;
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        $existing_feedback = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table_name} WHERE order_id = %d",
            $token_data['order_id']
        ));

        if ($existing_feedback) {
            wp_send_json_error('Feedback has already been submitted for this order');
        }

        $order = wc_get_order($token_data['order_id']);
        $customer_name = $order ? $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() : 'Unknown';

        // Validate required rating fields exist
        if (!isset($_POST['buying_process']) || !isset($_POST['page_rating']) || !isset($_POST['speed_rating'])) {
            wp_send_json_error('All rating fields are required');
        }

        // Enhanced input validation beyond sanitization
        $buying_process = intval($_POST['buying_process']);
        $page_rating = intval($_POST['page_rating']);
        $speed_rating = intval($_POST['speed_rating']);
        $additional_comments = isset($_POST['additional_comments']) ? sanitize_textarea_field($_POST['additional_comments']) : '';
        
        // Business logic validation
        $validation_errors = array();

        if ($buying_process < self::MIN_RATING_VALUE || $buying_process > self::MAX_RATING_VALUE) {
            $validation_errors[] = sprintf('Buying process rating must be between %d and %d', self::MIN_RATING_VALUE, self::MAX_RATING_VALUE);
        }

        if ($page_rating < self::MIN_RATING_VALUE || $page_rating > self::MAX_RATING_VALUE) {
            $validation_errors[] = sprintf('Page rating must be between %d and %d', self::MIN_RATING_VALUE, self::MAX_RATING_VALUE);
        }

        if ($speed_rating < self::MIN_RATING_VALUE || $speed_rating > self::MAX_RATING_VALUE) {
            $validation_errors[] = sprintf('Speed rating must be between %d and %d', self::MIN_RATING_VALUE, self::MAX_RATING_VALUE);
        }

        if (strlen($additional_comments) > self::MAX_COMMENT_LENGTH) {
            $validation_errors[] = sprintf('Comments must be less than %d characters', self::MAX_COMMENT_LENGTH);
        }
        
        // Check for suspicious patterns (basic spam detection)
        if (preg_match('/(http|ftp|www\.)/i', $additional_comments)) {
            $validation_errors[] = 'Comments cannot contain URLs';
        }
        
        if (!empty($validation_errors)) {
            wp_send_json_error('Validation failed: ' . implode(', ', $validation_errors));
        }
        
        $feedback_data = array(
            'order_id' => $token_data['order_id'],
            'customer_email' => $token_data['customer_email'],
            'customer_name' => $customer_name,
            'buying_process' => $buying_process,
            'page_rating' => $page_rating,
            'speed_rating' => $speed_rating,
            'additional_comments' => $additional_comments,
            'submitted_at' => current_time('mysql')
        );
        
        $feedback_id = $this->save_feedback_locally($feedback_data);
        
        // Audit log feedback submission
        $this->log_security_event('feedback_submitted', array(
            'feedback_id' => $feedback_id,
            'order_id' => $token_data['order_id'],
            'token_used' => substr($token, 0, 8) . '...'
        ));
        
        $notion_page_id = $this->send_to_notion($feedback_data, $feedback_id);
        
        if ($notion_page_id) {
            $this->log_security_event('notion_sync_success', array(
                'feedback_id' => $feedback_id,
                'notion_page_id' => substr($notion_page_id, 0, 8) . '...'
            ));
        } else {
            $this->log_security_event('notion_sync_failed', array(
                'feedback_id' => $feedback_id
            ));
        }
        
        delete_option("cftn_token_{$token}");

        // Get custom success message or use default
        $success_message = get_option('cftn_success_message', 'Thank you for your feedback!');

        // Check if Google review redirect is enabled
        $redirect_enabled = get_option('cftn_redirect_to_google_review', '0') === '1';
        $google_review_url = get_option('cftn_google_review_url', '');
        $redirect_delay = intval(get_option('cftn_redirect_delay', '3'));

        // Send response with redirect info
        wp_send_json_success(array(
            'message' => $success_message,
            'redirect_enabled' => $redirect_enabled,
            'redirect_url' => $google_review_url,
            'redirect_delay' => $redirect_delay
        ));
    }
    
    private function save_feedback_locally($feedback_data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        
        $wpdb->insert(
            $table_name,
            array(
                'order_id' => $feedback_data['order_id'],
                'customer_email' => $feedback_data['customer_email'],
                'customer_name' => $feedback_data['customer_name'],
                'buying_process_rating' => $feedback_data['buying_process'],
                'page_rating' => $feedback_data['page_rating'],
                'speed_rating' => $feedback_data['speed_rating'],
                'additional_comments' => $feedback_data['additional_comments'],
                'submitted_at' => $feedback_data['submitted_at']
            ),
            array('%d', '%s', '%s', '%d', '%d', '%d', '%s', '%s')
        );
        
        return $wpdb->insert_id;
    }
    
    private function send_to_notion($feedback_data, $feedback_id = null) {
        $notion_token = get_option('cftn_notion_token');
        $database_id = get_option('cftn_notion_database_id');

        if (!$notion_token || !$database_id) {
            return false;
        }

        // Check if this feedback is already synced (prevent duplicate syncs)
        if ($feedback_id) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'customer_feedback_notion';
            $is_synced = $wpdb->get_var($wpdb->prepare(
                "SELECT notion_synced FROM {$table_name} WHERE id = %d",
                $feedback_id
            ));

            if ($is_synced == 1) {
                // Already synced, return existing page ID
                $existing_page_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT notion_page_id FROM {$table_name} WHERE id = %d",
                    $feedback_id
                ));
                return $existing_page_id ?: true;
            }
        }
        
        // Ensure proper data types for Notion API
        $properties = array(
            'Name' => array(
                'title' => array(
                    array('text' => array('content' => strval($feedback_data['customer_name'])))
                )
            ),
            'Customer Name' => array(
                'rich_text' => array(
                    array('text' => array('content' => strval($feedback_data['customer_name'])))
                )
            ),
            'Email' => array(
                'email' => strval($feedback_data['customer_email'])
            ),
            'Order ID' => array(
                'number' => intval($feedback_data['order_id'])  // Must be integer, not string
            ),
            'Buying Process Rating' => array(
                'number' => intval($feedback_data['buying_process'])
            ),
            'Page Rating' => array(
                'number' => intval($feedback_data['page_rating'])
            ),
            'Speed Rating' => array(
                'number' => intval($feedback_data['speed_rating'])
            ),
            'Submitted Date' => array(
                'date' => array(
                    'start' => date('Y-m-d', strtotime($feedback_data['submitted_at']))
                )
            )
        );

        if (!empty($feedback_data['additional_comments'])) {
            // Notion rich_text has a character limit per block
            $comment_text = substr($feedback_data['additional_comments'], 0, self::MAX_COMMENT_LENGTH);
            $properties['Comments'] = array(
                'rich_text' => array(
                    array('text' => array('content' => $comment_text))
                )
            );
        }
        
        $body = array(
            'parent' => array('database_id' => $database_id),
            'properties' => $properties
        );

        $response = wp_remote_post('https://api.notion.com/v1/pages', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $notion_token,
                'Content-Type' => 'application/json',
                'Notion-Version' => '2022-06-28'
            ),
            'body' => json_encode($body),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            $error_message = 'WP_Error: ' . $response->get_error_message() . ' (Code: ' . $response->get_error_code() . ')';

            // Store detailed error for admin display
            update_option('cftn_last_notion_error', array(
                'time' => current_time('mysql'),
                'type' => 'connection_error',
                'message' => $error_message
            ));

            return false;
        } else {
            $status_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);

            if ($status_code === 200) {
                $response_data = json_decode($response_body, true);
                $notion_page_id = $response_data['id'] ?? null;

                if ($feedback_id && $notion_page_id) {
                    $this->update_feedback_sync_status($feedback_id, $notion_page_id);
                }

                // Clear any previous errors on success
                delete_option('cftn_last_notion_error');

                return $notion_page_id;
            } else {
                // Parse error response for better debugging and user feedback
                $error_data = json_decode($response_body, true);
                $user_friendly_message = $this->get_notion_error_message($status_code, $error_data);

                // Store detailed error for admin display
                update_option('cftn_last_notion_error', array(
                    'time' => current_time('mysql'),
                    'type' => 'api_error',
                    'status_code' => $status_code,
                    'message' => $user_friendly_message,
                    'raw_response' => $response_body
                ));

                return false;
            }
        }
    }
    
    /**
     * Get user-friendly error message from Notion API response
     */
    private function get_notion_error_message($status_code, $error_data) {
        $message = 'Notion API Error (Status ' . $status_code . '): ';

        // Common Notion API errors with helpful messages
        switch ($status_code) {
            case 400:
                if (isset($error_data['code']) && $error_data['code'] === 'validation_error') {
                    $message .= 'Validation error - Check that your Notion database fields match the expected structure. ';
                    if (isset($error_data['message'])) {
                        $message .= 'Details: ' . $error_data['message'];
                    }
                } else {
                    $message .= 'Bad request - Invalid data sent to Notion.';
                }
                break;

            case 401:
                $message .= 'Unauthorized - Your Notion API token is invalid or expired. Please update it in settings.';
                break;

            case 403:
                $message .= 'Forbidden - The integration doesn\'t have access to this database. Make sure you\'ve shared the database with your integration in Notion.';
                break;

            case 404:
                $message .= 'Not found - The database ID is incorrect or the database has been deleted.';
                break;

            case 429:
                $message .= 'Rate limited - Too many requests. Please wait a moment and try again.';
                break;

            case 500:
            case 502:
            case 503:
                $message .= 'Notion server error - This is a temporary issue on Notion\'s side. Please try again later.';
                break;

            default:
                $message .= 'Unknown error.';
                if (isset($error_data['message'])) {
                    $message .= ' ' . $error_data['message'];
                }
        }

        return $message;
    }

    private function update_feedback_sync_status($feedback_id, $notion_page_id) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'customer_feedback_notion';

        // Only update if not already synced (prevent double-sync)
        $wpdb->update(
            $table_name,
            array(
                'notion_synced' => 1,
                'notion_page_id' => $notion_page_id
            ),
            array(
                'id' => $feedback_id,
                'notion_synced' => 0  // Only update if not already synced
            ),
            array('%d', '%s'),
            array('%d', '%d')
        );
    }
    
    public function add_admin_menu() {
        $main_page = add_menu_page(
            'Customer Feedback to Notion',
            'Feedback to Notion',
            'manage_options',
            'customer-feedback-notion',
            array($this, 'dashboard_page'),
            'dashicons-feedback',
            25
        );
        
        add_submenu_page(
            'customer-feedback-notion',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'customer-feedback-notion',
            array($this, 'dashboard_page')
        );
        
        add_submenu_page(
            'customer-feedback-notion',
            'Settings',
            'Settings',
            'manage_options',
            'customer-feedback-notion-settings',
            array($this, 'settings_page')
        );
        
        add_submenu_page(
            'customer-feedback-notion',
            'Analytics',
            'Analytics',
            'manage_options',
            'customer-feedback-notion-analytics',
            array($this, 'analytics_page')
        );
    }
    
    public function settings_init() {
        register_setting('cftn_settings', 'cftn_notion_token', array(
            'sanitize_callback' => array($this, 'sanitize_notion_token')
        ));
        register_setting('cftn_settings', 'cftn_notion_database_id', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('cftn_settings', 'cftn_success_message', array(
            'sanitize_callback' => 'wp_kses_post'
        ));
        register_setting('cftn_settings', 'cftn_redirect_to_google_review', array(
            'sanitize_callback' => array($this, 'sanitize_checkbox')
        ));
        register_setting('cftn_settings', 'cftn_google_review_url', array(
            'sanitize_callback' => array($this, 'sanitize_google_review_url')
        ));
        register_setting('cftn_settings', 'cftn_redirect_delay', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('cftn_settings', 'cftn_email_subject', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('cftn_settings', 'cftn_email_body', array(
            'sanitize_callback' => 'wp_kses_post'
        ));
        register_setting('cftn_settings', 'cftn_token_expiry_days', array(
            'sanitize_callback' => 'absint'
        ));

        add_settings_section(
            'cftn_settings_section',
            __('Notion Integration Settings', 'customer-feedback-to-notion'),
            null,
            'cftn_settings'
        );

        add_settings_field(
            'cftn_notion_token',
            __('Notion API Token', 'customer-feedback-to-notion'),
            array($this, 'notion_token_field'),
            'cftn_settings',
            'cftn_settings_section'
        );

        add_settings_field(
            'cftn_notion_database_id',
            __('Notion Database ID', 'customer-feedback-to-notion'),
            array($this, 'notion_database_field'),
            'cftn_settings',
            'cftn_settings_section'
        );

        add_settings_section(
            'cftn_email_section',
            __('Email Customization', 'customer-feedback-to-notion'),
            array($this, 'email_section_callback'),
            'cftn_settings'
        );

        add_settings_field(
            'cftn_email_subject',
            __('Email Subject', 'customer-feedback-to-notion'),
            array($this, 'email_subject_field'),
            'cftn_settings',
            'cftn_email_section'
        );

        add_settings_field(
            'cftn_email_body',
            __('Email Body', 'customer-feedback-to-notion'),
            array($this, 'email_body_field'),
            'cftn_settings',
            'cftn_email_section'
        );

        add_settings_field(
            'cftn_token_expiry_days',
            __('Feedback Link Expiry (Days)', 'customer-feedback-to-notion'),
            array($this, 'token_expiry_days_field'),
            'cftn_settings',
            'cftn_email_section'
        );

        add_settings_section(
            'cftn_messages_section',
            __('Message Customization', 'customer-feedback-to-notion'),
            null,
            'cftn_settings'
        );

        add_settings_field(
            'cftn_success_message',
            __('Success Message', 'customer-feedback-to-notion'),
            array($this, 'success_message_field'),
            'cftn_settings',
            'cftn_messages_section'
        );

        add_settings_section(
            'cftn_google_review_section',
            __('Google Review Integration', 'customer-feedback-to-notion'),
            array($this, 'google_review_section_callback'),
            'cftn_settings'
        );

        add_settings_field(
            'cftn_redirect_to_google_review',
            __('Redirect to Google Review', 'customer-feedback-to-notion'),
            array($this, 'redirect_to_google_field'),
            'cftn_settings',
            'cftn_google_review_section'
        );

        add_settings_field(
            'cftn_google_review_url',
            __('Google Review URL', 'customer-feedback-to-notion'),
            array($this, 'google_review_url_field'),
            'cftn_settings',
            'cftn_google_review_section'
        );

        add_settings_field(
            'cftn_redirect_delay',
            __('Redirect Delay (seconds)', 'customer-feedback-to-notion'),
            array($this, 'redirect_delay_field'),
            'cftn_settings',
            'cftn_google_review_section'
        );
    }
    
    public function sanitize_checkbox($input) {
        return ($input === '1' || $input === 1 || $input === true) ? '1' : '0';
    }

    public function sanitize_notion_token($input) {
        $input = sanitize_text_field($input);

        // If input is empty or contains only asterisks (masked value), keep existing token
        if (empty($input) || preg_match('/^\*+/', $input)) {
            return get_option('cftn_notion_token');
        }

        return $input;
    }

    public function sanitize_google_review_url($input) {
        $input = esc_url_raw($input);

        // If empty, allow it (redirect is optional)
        if (empty($input)) {
            return '';
        }

        // Whitelist allowed domains for Google review URLs
        $allowed_domains = array('google.com', 'g.page', 'maps.google.com', 'goo.gl');
        $parsed_url = parse_url($input);

        if (!isset($parsed_url['host'])) {
            add_settings_error('cftn_google_review_url', 'invalid_url', __('Invalid URL format', 'customer-feedback-to-notion'));
            return '';
        }

        $host = strtolower($parsed_url['host']);
        $is_allowed = false;

        foreach ($allowed_domains as $domain) {
            if ($host === $domain || substr($host, -(strlen($domain) + 1)) === '.' . $domain) {
                $is_allowed = true;
                break;
            }
        }

        if (!$is_allowed) {
            add_settings_error('cftn_google_review_url', 'invalid_domain', __('Only Google review URLs are allowed (google.com, g.page, maps.google.com, goo.gl)', 'customer-feedback-to-notion'));
            return '';
        }

        return $input;
    }

    public function notion_token_field() {
        $value = get_option('cftn_notion_token');
        // Mask token for security - only show last 4 characters
        $masked_value = '';
        if (!empty($value)) {
            $masked_value = str_repeat('*', max(0, strlen($value) - 4)) . substr($value, -4);
        }
        ?>
        <input type="text" name="cftn_notion_token" value="<?php echo esc_attr($masked_value); ?>"
               style="width: 400px;"
               placeholder="<?php echo $value ? __('Token is set (masked for security)', 'customer-feedback-to-notion') : __('Enter your Notion API token', 'customer-feedback-to-notion'); ?>"
               onfocus="if(this.value.indexOf('*')===0){this.value='';this.type='password';}" />
        <p class="description">
            <?php _e('Your Notion integration API token. Get it from <a href="https://www.notion.so/my-integrations" target="_blank">https://www.notion.so/my-integrations</a>', 'customer-feedback-to-notion'); ?>
            <?php if ($value): ?>
                <br><strong><?php _e('Note: Leave blank to keep existing token, or enter a new token to replace it.', 'customer-feedback-to-notion'); ?></strong>
            <?php endif; ?>
        </p>
        <?php
    }
    
    public function notion_database_field() {
        $value = get_option('cftn_notion_database_id');
        echo '<input type="text" name="cftn_notion_database_id" value="' . esc_attr($value) . '" style="width: 400px;" />';
        echo '<p class="description">' . __('The ID of your Notion database where feedback will be stored. Found in the database URL.', 'customer-feedback-to-notion') . '</p>';
    }

    public function email_section_callback() {
        echo '<p>' . __('Customize the feedback request email sent to customers when their order is completed.', 'customer-feedback-to-notion') . '</p>';
        echo '<p><strong>' . __('Available placeholders:', 'customer-feedback-to-notion') . '</strong></p>';
        echo '<ul style="list-style-type: disc; margin-left: 20px;">';
        echo '<li><code>{customer_name}</code> - ' . __('Customer first name', 'customer-feedback-to-notion') . '</li>';
        echo '<li><code>{order_number}</code> - ' . __('Order number', 'customer-feedback-to-notion') . '</li>';
        echo '<li><code>{feedback_url}</code> - ' . __('Feedback form link', 'customer-feedback-to-notion') . '</li>';
        echo '<li><code>{expiry_days}</code> - ' . __('Number of days until link expires', 'customer-feedback-to-notion') . '</li>';
        echo '</ul>';
    }

    public function email_subject_field() {
        $default_subject = __('How was your shopping experience?', 'customer-feedback-to-notion');
        $value = get_option('cftn_email_subject', $default_subject);
        echo '<input type="text" name="cftn_email_subject" value="' . esc_attr($value) . '" style="width: 500px;" />';
        echo '<p class="description">' . __('Subject line for the feedback request email.', 'customer-feedback-to-notion') . '</p>';
    }

    public function email_body_field() {
        $default_body = "Hi {customer_name},\n\nThank you for your recent order #{order_number}!\n\nWe'd love to hear about your shopping experience. Please take a moment to share your feedback:\n\n{feedback_url}\n\nThis link will expire in {expiry_days} days.\n\nThank you!";
        $value = get_option('cftn_email_body', $default_body);
        echo '<textarea name="cftn_email_body" rows="10" style="width: 500px; font-family: monospace;">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">' . __('Email body text. Use the placeholders above to personalize the message.', 'customer-feedback-to-notion') . '</p>';
    }

    public function token_expiry_days_field() {
        $value = get_option('cftn_token_expiry_days', self::TOKEN_EXPIRY_DAYS);
        echo '<input type="number" name="cftn_token_expiry_days" value="' . esc_attr($value) . '" min="1" max="30" style="width: 100px;" />';
        echo '<p class="description">' . __('Number of days the feedback link remains valid (1-30 days). Default: 5 days.', 'customer-feedback-to-notion') . '</p>';
    }

    public function success_message_field() {
        $value = get_option('cftn_success_message', 'Thank you for your feedback!');
        echo '<textarea name="cftn_success_message" rows="3" style="width: 400px;">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">' . __('Message shown to customers after they successfully submit feedback. You can use safe HTML tags (links, bold, italic, etc.).', 'customer-feedback-to-notion') . '</p>';
    }

    public function google_review_section_callback() {
        echo '<p>' . __('After customers submit feedback, you can optionally redirect them to leave a Google review.', 'customer-feedback-to-notion') . '</p>';
    }

    public function redirect_to_google_field() {
        $value = get_option('cftn_redirect_to_google_review', '0');
        ?>
        <label>
            <input type="checkbox" name="cftn_redirect_to_google_review" value="1" <?php checked($value, '1'); ?> />
            <?php _e('Enable redirect to Google Business review page after feedback submission', 'customer-feedback-to-notion'); ?>
        </label>
        <?php
    }

    public function google_review_url_field() {
        $value = get_option('cftn_google_review_url', '');
        echo '<input type="url" name="cftn_google_review_url" value="' . esc_attr($value) . '" style="width: 400px;" placeholder="https://g.page/r/..." />';
        echo '<p class="description">' . __('Your Google Business review URL. Find it by going to your Google Business Profile → Get more reviews → Copy short URL.', 'customer-feedback-to-notion') . '<br>';
        echo __('Example: <code>https://g.page/r/CaBcDeFgHiJkLmN/review</code>', 'customer-feedback-to-notion') . '</p>';
    }

    public function redirect_delay_field() {
        $value = get_option('cftn_redirect_delay', '3');
        echo '<input type="number" name="cftn_redirect_delay" value="' . esc_attr($value) . '" min="0" max="' . self::MAX_REDIRECT_DELAY . '" style="width: 100px;" />';
        echo '<p class="description">' . sprintf(__('How many seconds to wait before redirecting (0-%d). Default: 3 seconds.', 'customer-feedback-to-notion'), self::MAX_REDIRECT_DELAY) . '</p>';
    }
    
    private function get_feedback_stats() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'customer_feedback_notion';

        // Direct queries are safe when table name is constructed from wpdb->prefix
        $total = $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}`");

        $this_month = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table_name}` WHERE submitted_at >= %s",
            date('Y-m-01')
        ));

        $avg_rating = $wpdb->get_var(
            "SELECT AVG((buying_process_rating + page_rating + speed_rating) / 3) FROM `{$table_name}`"
        );

        $synced = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table_name}` WHERE notion_synced = %d",
            1
        ));

        return array(
            'total' => intval($total),
            'this_month' => intval($this_month),
            'avg_rating' => floatval($avg_rating),
            'synced' => intval($synced)
        );
    }
    
    private function get_recent_feedback($limit = 10) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_name} ORDER BY submitted_at DESC LIMIT %d",
            $limit
        ));
    }
    
    private function get_feedback_analytics() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'customer_feedback_notion';

        // Direct queries are safe when table name is constructed from wpdb->prefix
        $total = $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}`");

        $buying_process = array();
        $page_rating = array();
        $speed_rating = array();

        for ($i = 1; $i <= 5; $i++) {
            $buying_process[$i] = intval($wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name}` WHERE buying_process_rating = %d",
                $i
            )));

            $page_rating[$i] = intval($wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name}` WHERE page_rating = %d",
                $i
            )));

            $speed_rating[$i] = intval($wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name}` WHERE speed_rating = %d",
                $i
            )));
        }

        $monthly_trend = array();
        // Table name is safe (from wpdb->prefix), no user input
        $months = $wpdb->get_results(
            "SELECT DATE_FORMAT(submitted_at, '%Y-%m') as month, COUNT(*) as count
             FROM `{$table_name}`
             WHERE submitted_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
             GROUP BY month
             ORDER BY month"
        );

        foreach ($months as $month_data) {
            $monthly_trend[$month_data->month] = intval($month_data->count);
        }

        return array(
            'total' => intval($total),
            'buying_process' => $buying_process,
            'page_rating' => $page_rating,
            'speed_rating' => $speed_rating,
            'monthly_trend' => $monthly_trend
        );
    }
    
    public function test_notion_connection() {
        // Check both nonce and user capabilities
        if (!wp_verify_nonce($_POST['nonce'], 'cftn_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $notion_token = get_option('cftn_notion_token');
        $database_id = get_option('cftn_notion_database_id');

        if (!$notion_token || !$database_id) {
            wp_send_json_error('Notion API token and database ID are required.');
        }

        $response = wp_remote_get("https://api.notion.com/v1/databases/{$database_id}", array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $notion_token,
                'Notion-Version' => '2022-06-28'
            ),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            wp_send_json_error('Connection error: ' . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code === 200) {
            $body = wp_remote_retrieve_body($response);
            $database_info = json_decode($body, true);

            // Validate database schema
            $validation = $this->validate_notion_database_schema($database_info);

            if ($validation['valid']) {
                wp_send_json_success('✅ Connection successful! Database schema is correctly configured.');
            } else {
                wp_send_json_error('⚠️ Connection successful, but database schema issues found:<br>' . implode('<br>', $validation['errors']));
            }
        } else {
            $body = wp_remote_retrieve_body($response);
            wp_send_json_error('API error (Status: ' . $status_code . '): ' . $body);
        }
    }

    /**
     * Validate that the Notion database has the required fields with correct types
     */
    private function validate_notion_database_schema($database_info) {
        $required_fields = array(
            'Name' => 'title',  // Every Notion database needs a title field
            'Customer Name' => 'rich_text',
            'Email' => 'email',
            'Order ID' => 'number',
            'Buying Process Rating' => 'number',
            'Page Rating' => 'number',
            'Speed Rating' => 'number',
            'Submitted Date' => 'date',
            'Comments' => 'rich_text'  // Optional
        );

        $errors = array();
        $properties = $database_info['properties'] ?? array();

        foreach ($required_fields as $field_name => $expected_type) {
            if ($field_name === 'Comments') {
                // Comments is optional
                continue;
            }

            if (!isset($properties[$field_name])) {
                $errors[] = "Missing field: '<strong>{$field_name}</strong>'";
                continue;
            }

            $actual_type = $properties[$field_name]['type'] ?? 'unknown';
            if ($actual_type !== $expected_type) {
                $errors[] = "Field '<strong>{$field_name}</strong>' has type '<strong>{$actual_type}</strong>' but should be '<strong>{$expected_type}</strong>'";
            }
        }

        return array(
            'valid' => empty($errors),
            'errors' => $errors
        );
    }
    
    /**
     * Clear the last error message
     */
    public function clear_last_error() {
        if (!wp_verify_nonce($_POST['nonce'], 'cftn_clear_error')) {
            wp_send_json_error('Security check failed');
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        delete_option('cftn_last_notion_error');
        wp_send_json_success('Error cleared');
    }

    public function sync_feedback_data() {
        // Check both nonce and user capabilities
        if (!wp_verify_nonce($_POST['nonce'], 'cftn_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        global $wpdb;
        
        $table_name = $wpdb->prefix . 'customer_feedback_notion';
        
        $unsynced_feedback = $wpdb->get_results(
            "SELECT * FROM {$table_name} WHERE notion_synced = 0 ORDER BY submitted_at DESC LIMIT 10"
        );
        
        if (empty($unsynced_feedback)) {
            wp_send_json_success('All feedback is already synced!');
        }
        
        $synced_count = 0;
        $errors = array();

        foreach ($unsynced_feedback as $feedback) {
            $feedback_data = array(
                'order_id' => $feedback->order_id,
                'customer_email' => $feedback->customer_email,
                'customer_name' => $feedback->customer_name,
                'buying_process' => $feedback->buying_process_rating,
                'page_rating' => $feedback->page_rating,
                'speed_rating' => $feedback->speed_rating,
                'additional_comments' => $feedback->additional_comments,
                'submitted_at' => $feedback->submitted_at
            );

            $notion_page_id = $this->send_to_notion($feedback_data, $feedback->id);

            if ($notion_page_id) {
                $synced_count++;
            } else {
                // Get the detailed error from the last sync attempt
                $last_error = get_option('cftn_last_notion_error');
                $error_detail = '';

                if ($last_error && isset($last_error['message'])) {
                    $error_detail = ' - ' . $last_error['message'];
                }

                $errors[] = "Order #{$feedback->order_id}{$error_detail}";
            }
        }

        if ($synced_count > 0 && !empty($errors)) {
            // Partial success
            wp_send_json_success(sprintf('Synced %d of %d feedback entries. Errors: %s', $synced_count, count($unsynced_feedback), implode(' | ', $errors)));
        } elseif ($synced_count > 0) {
            wp_send_json_success(sprintf('Successfully synced %d feedback entries.', $synced_count));
        } else {
            wp_send_json_error('Failed to sync feedback:<br><br>' . implode('<br>', $errors));
        }
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'customer-feedback-notion') !== false) {
            wp_enqueue_script('cftn-admin', plugin_dir_url(__FILE__) . 'assets/admin.js', array('jquery'), '1.0.0', true);
            wp_localize_script('cftn-admin', 'cftn_admin_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('cftn_admin_nonce')
            ));
            wp_enqueue_style('cftn-admin', plugin_dir_url(__FILE__) . 'assets/admin.css', array(), '1.0.0');
        }
    }
    
    public function dashboard_page() {
        $stats = $this->get_feedback_stats();
        $recent_feedback = $this->get_recent_feedback(10);
        $last_error = get_option('cftn_last_notion_error');
        ?>
        <div class="wrap">
            <h1><?php _e('Customer Feedback Dashboard', 'customer-feedback-to-notion'); ?></h1>

            <div class="notice notice-info" style="margin-top: 20px; padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1;">
                <p style="margin: 0;">
                    <strong><?php _e('Plugin by COOD3', 'customer-feedback-to-notion'); ?></strong> -
                    <a href="<?php echo esc_url('https://cood3.com/en/cftn/'); ?>" target="_blank" rel="noopener noreferrer">
                        <?php _e('Visit Plugin Homepage', 'customer-feedback-to-notion'); ?> ↗
                    </a>
                    <span style="margin-left: 15px; opacity: 0.7;">|</span>
                    <a href="<?php echo esc_url('https://cood3.com/en/cftn_documentation'); ?>" target="_blank" rel="noopener noreferrer" style="margin-left: 15px;">
                        <?php _e('Documentation', 'customer-feedback-to-notion'); ?> ↗
                    </a>
                    <span style="margin-left: 15px; opacity: 0.7;">|</span>
                    <a href="<?php echo esc_url('https://cood3.com/en/cftn_documentation/#support'); ?>" target="_blank" rel="noopener noreferrer" style="margin-left: 15px;">
                        <?php _e('Support', 'customer-feedback-to-notion'); ?> ↗
                    </a>
                    <span style="margin-left: 15px; opacity: 0.7;">|</span>
                    <a href="<?php echo esc_url('https://cood3.com/wp-content/uploads/2025/09/cftn_103.zip'); ?>" target="_blank" rel="noopener noreferrer" style="margin-left: 15px;">
                        <?php _e('Download Plugin', 'customer-feedback-to-notion'); ?> ↗
                    </a>
                </p>
            </div>

            <?php if ($last_error): ?>
                <div class="notice notice-error">
                    <p>
                        <strong><?php _e('Last Notion Sync Error:', 'customer-feedback-to-notion'); ?></strong><br>
                        <?php echo nl2br(esc_html($last_error['message'])); ?>
                    </p>
                    <p><small><?php echo esc_html($last_error['time']); ?></small></p>
                    <?php if (isset($last_error['raw_response']) && defined('WP_DEBUG') && WP_DEBUG): ?>
                        <details>
                            <summary style="cursor: pointer; font-weight: bold;">Show Raw Response (Debug)</summary>
                            <pre style="background: #f5f5f5; padding: 10px; overflow: auto; max-height: 300px;"><?php echo esc_html($last_error['raw_response']); ?></pre>
                        </details>
                    <?php endif; ?>
                    <p>
                        <button type="button" class="button" onclick="jQuery(this).closest('.notice').remove(); jQuery.post(ajaxurl, {action: 'cftn_clear_error', nonce: '<?php echo wp_create_nonce('cftn_clear_error'); ?>'});">
                            <?php _e('Dismiss', 'customer-feedback-to-notion'); ?>
                        </button>
                    </p>
                </div>
            <?php endif; ?>

            <div class="cftn-dashboard">
                <div class="cftn-stats-row">
                    <div class="cftn-stat-card">
                        <h3><?php _e('Total Feedback', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-stat-number"><?php echo esc_html($stats['total']); ?></div>
                    </div>
                    
                    <div class="cftn-stat-card">
                        <h3><?php _e('This Month', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-stat-number"><?php echo esc_html($stats['this_month']); ?></div>
                    </div>
                    
                    <div class="cftn-stat-card">
                        <h3><?php _e('Average Rating', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-stat-number"><?php echo number_format($stats['avg_rating'], 1); ?>/5</div>
                    </div>
                    
                    <div class="cftn-stat-card">
                        <h3><?php _e('Notion Synced', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-stat-number"><?php echo esc_html($stats['synced']); ?>/<span class="total"><?php echo esc_html($stats['total']); ?></span></div>
                    </div>
                </div>
                
                <div class="cftn-actions">
                    <button id="cftn-test-connection" class="button button-secondary"><?php _e('Test Notion Connection', 'customer-feedback-to-notion'); ?></button>
                    <button id="cftn-sync-data" class="button button-primary"><?php _e('Sync Unsynced Data', 'customer-feedback-to-notion'); ?></button>
                    <a href="<?php echo admin_url('admin.php?page=customer-feedback-notion-settings'); ?>" class="button"><?php _e('Settings', 'customer-feedback-to-notion'); ?></a>
                </div>
                
                <h2><?php _e('Recent Feedback', 'customer-feedback-to-notion'); ?></h2>
                <div class="cftn-feedback-table">
                    <?php if (empty($recent_feedback)): ?>
                        <p><?php _e('No feedback received yet.', 'customer-feedback-to-notion'); ?></p>
                    <?php else: ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Customer', 'customer-feedback-to-notion'); ?></th>
                                    <th><?php _e('Order ID', 'customer-feedback-to-notion'); ?></th>
                                    <th><?php _e('Ratings', 'customer-feedback-to-notion'); ?></th>
                                    <th><?php _e('Comments', 'customer-feedback-to-notion'); ?></th>
                                    <th><?php _e('Date', 'customer-feedback-to-notion'); ?></th>
                                    <th><?php _e('Notion', 'customer-feedback-to-notion'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_feedback as $feedback): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html($feedback->customer_name); ?></strong><br>
                                        <small><?php echo esc_html($feedback->customer_email); ?></small>
                                    </td>
                                    <td><?php echo esc_html($feedback->order_id); ?></td>
                                    <td>
                                        <div class="cftn-ratings">
                                            <span title="<?php _e('Buying Process', 'customer-feedback-to-notion'); ?>"><?php echo str_repeat('⭐', $feedback->buying_process_rating); ?></span><br>
                                            <span title="<?php _e('Page Rating', 'customer-feedback-to-notion'); ?>"><?php echo str_repeat('⭐', $feedback->page_rating); ?></span><br>
                                            <span title="<?php _e('Speed Rating', 'customer-feedback-to-notion'); ?>"><?php echo str_repeat('⭐', $feedback->speed_rating); ?></span>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ($feedback->additional_comments): ?>
                                            <div class="cftn-comment" title="<?php echo esc_attr($feedback->additional_comments); ?>">
                                                <?php echo esc_html(wp_trim_words($feedback->additional_comments, 10)); ?>
                                            </div>
                                        <?php else: ?>
                                            <em><?php _e('No comments', 'customer-feedback-to-notion'); ?></em>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($feedback->submitted_at)); ?></td>
                                    <td>
                                        <?php if ($feedback->notion_synced): ?>
                                            <span class="cftn-status synced">✓ <?php _e('Synced', 'customer-feedback-to-notion'); ?></span>
                                        <?php else: ?>
                                            <span class="cftn-status pending">⏳ <?php _e('Pending', 'customer-feedback-to-notion'); ?></span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Customer Feedback to Notion Settings', 'customer-feedback-to-notion'); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('cftn_settings');
                do_settings_sections('cftn_settings');
                submit_button();
                ?>
            </form>
            
            <h2><?php _e('Setup Instructions', 'customer-feedback-to-notion'); ?></h2>
            <ol>
                <li><?php _e('Create a Notion integration at https://www.notion.so/my-integrations', 'customer-feedback-to-notion'); ?></li>
                <li><?php _e('Copy the API token and paste it above', 'customer-feedback-to-notion'); ?></li>
                <li><?php _e('Create a database in Notion with these EXACT properties (names and types must match):', 'customer-feedback-to-notion'); ?>
                    <table class="wp-list-table widefat" style="max-width: 600px; margin-top: 10px;">
                        <thead>
                            <tr>
                                <th><strong>Property Name</strong></th>
                                <th><strong>Property Type</strong></th>
                                <th><strong>Required</strong></th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr><td><code>Name</code></td><td>Title</td><td>✅ Yes (default title field)</td></tr>
                            <tr><td><code>Customer Name</code></td><td>Text</td><td>✅ Yes</td></tr>
                            <tr><td><code>Email</code></td><td>Email</td><td>✅ Yes</td></tr>
                            <tr><td><code>Order ID</code></td><td>Number</td><td>✅ Yes</td></tr>
                            <tr><td><code>Buying Process Rating</code></td><td>Number</td><td>✅ Yes</td></tr>
                            <tr><td><code>Page Rating</code></td><td>Number</td><td>✅ Yes</td></tr>
                            <tr><td><code>Speed Rating</code></td><td>Number</td><td>✅ Yes</td></tr>
                            <tr><td><code>Submitted Date</code></td><td>Date</td><td>✅ Yes</td></tr>
                            <tr><td><code>Comments</code></td><td>Text</td><td>⚪ Optional</td></tr>
                        </tbody>
                    </table>
                    <p class="description" style="margin-top: 10px;">
                        <strong>⚠️ Important:</strong> Property names are case-sensitive and must match exactly (including spaces).
                        The default "Name" field should remain as Title type. "Customer Name" should be Text type.<br>
                        Use the "Test Connection" button on the Dashboard to verify your database structure.
                    </p>
                </li>
                <li><?php _e('Share the database with your integration (click "..." → "Connections" → Add your integration)', 'customer-feedback-to-notion'); ?></li>
                <li><?php _e('Copy the database ID from the URL (the part after the workspace name and before the "?")', 'customer-feedback-to-notion'); ?>
                    <p class="description">Example: <code>https://www.notion.so/myworkspace/<strong>a1b2c3d4e5f6</strong>?v=...</code></p>
                </li>
            </ol>
            
            <h2><?php _e('How it works', 'customer-feedback-to-notion'); ?></h2>
            <p><?php _e('When an order is completed, the customer receives an email with a feedback link. The feedback form collects ratings for the buying process, page quality, and website speed, then sends the data to your Notion database.', 'customer-feedback-to-notion'); ?></p>
        </div>
        <?php
    }
    
    public function analytics_page() {
        $analytics = $this->get_feedback_analytics();
        ?>
        <div class="wrap">
            <h1><?php _e('Feedback Analytics', 'customer-feedback-to-notion'); ?></h1>
            
            <div class="cftn-analytics">
                <h2><?php _e('Rating Breakdown', 'customer-feedback-to-notion'); ?></h2>
                <div class="cftn-rating-charts">
                    <div class="cftn-chart-container">
                        <h3><?php _e('Buying Process Ratings', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-bar-chart">
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <?php $count = $analytics['buying_process'][$i] ?? 0; ?>
                                <div class="cftn-bar-item">
                                    <span class="cftn-bar-label"><?php echo $i; ?>⭐</span>
                                    <div class="cftn-bar" style="width: <?php echo $analytics['total'] > 0 ? ($count / $analytics['total'] * 100) : 0; ?>%;"></div>
                                    <span class="cftn-bar-count"><?php echo $count; ?></span>
                                </div>
                            <?php endfor; ?>
                        </div>
                    </div>
                    
                    <div class="cftn-chart-container">
                        <h3><?php _e('Page Ratings', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-bar-chart">
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <?php $count = $analytics['page_rating'][$i] ?? 0; ?>
                                <div class="cftn-bar-item">
                                    <span class="cftn-bar-label"><?php echo $i; ?>⭐</span>
                                    <div class="cftn-bar" style="width: <?php echo $analytics['total'] > 0 ? ($count / $analytics['total'] * 100) : 0; ?>%;"></div>
                                    <span class="cftn-bar-count"><?php echo $count; ?></span>
                                </div>
                            <?php endfor; ?>
                        </div>
                    </div>
                    
                    <div class="cftn-chart-container">
                        <h3><?php _e('Speed Ratings', 'customer-feedback-to-notion'); ?></h3>
                        <div class="cftn-bar-chart">
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <?php $count = $analytics['speed_rating'][$i] ?? 0; ?>
                                <div class="cftn-bar-item">
                                    <span class="cftn-bar-label"><?php echo $i; ?>⭐</span>
                                    <div class="cftn-bar" style="width: <?php echo $analytics['total'] > 0 ? ($count / $analytics['total'] * 100) : 0; ?>%;"></div>
                                    <span class="cftn-bar-count"><?php echo $count; ?></span>
                                </div>
                            <?php endfor; ?>
                        </div>
                    </div>
                </div>
                
                <h2><?php _e('Monthly Feedback Trend', 'customer-feedback-to-notion'); ?></h2>
                <div class="cftn-trend-chart">
                    <?php foreach ($analytics['monthly_trend'] as $month => $count): ?>
                        <div class="cftn-trend-item">
                            <span class="cftn-trend-label"><?php echo date_i18n('M Y', strtotime($month . '-01')); ?></span>
                            <div class="cftn-trend-bar" style="height: <?php echo $count > 0 ? min(($count / max($analytics['monthly_trend']) * 100), 100) : 0; ?>%;"></div>
                            <span class="cftn-trend-count"><?php echo $count; ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }
}

// Initialize the plugin
CustomerFeedbackToNotion::get_instance();

register_deactivation_hook(__FILE__, 'cftn_deactivation');
function cftn_deactivation() {
    wp_clear_scheduled_hook('cftn_cleanup_expired_tokens');
}